/* nethacklaunch.c
 * (c)2001 M. Drew Streib. This program is released under the terms of the
 * GNU General Public License version 2.
 *
 * dtype@dtype.org
 * See this in action at http://dtype.org/nethack/
 *
 * This is a little wrapper for nethack (and soon other programs) that
 * will allow them to be run from a telnetd session, chroot, shed privs,
 * make a simple login, then play the game.
 *
 * By default, this thing is also statically compiled, and can thus be
 * run inside of a chroot jail itself if necessary.
 *
 * Yes, I know it is all global variables. Deal with it. The program
 * is very small.
 */

/* Changelog
 * 
 * 0.1.8
 *   Reverted changes for eterm.
 * 0.1.7
 *   Also added check for ERR at getch().
 * 0.1.6 
 *   Added exit() on any errors in refresh(), which should get rid
 *   of the hanging problem on disconnected clients.
 *
 * /

/* IMPORTANT defines */

#define SHED_UID 1031	/* the uid to shed privs to */
#define SHED_GID 1031	/* the gid to shed privs to */
#define MAXUSERS 4000	/* solves some preallocation issues. */

/* a request from the author: please leave some remnance of
 * 'based on dgamelaunch version xxx' in any derivative works, or
 * even keep the line the same altogether. I'm probably happy 
 * to make any changes you need. */

#define VER1 "## dgamelaunch - network console game launcher\n"
#define VER2 "## version 0.1.9\n"
#define SERVER "## Server info is at http://dtype.org/nethack/"

/* ************************************************************* */

/* program stuff */

#include <curses.h>
#include <sys/types.h>
#include <sys/file.h> /* for flock() */
#include <errno.h>

int loggedin = 0;
char my_name[21];
char my_email[81];
char my_pw[21];
char my_env[1025];

/* preallocate this mem. bad, but ohwell. is only for pointers */
/* makes a max number of users compiled in */
int f_num = 0;
char *f_name[MAXUSERS];
char *f_email[MAXUSERS];
char *f_pw[MAXUSERS];
char *f_env[MAXUSERS];

int loopcontrol1 = 0;
int loopcontrol2 = 0;

/* ************************************************************* */

void drawmenu() {
  char buf[80];

  clear();

  mvaddstr(1,1,VER1);
  mvaddstr(2,1,VER2);
  mvaddstr(3,1,SERVER);

  if(loggedin) {
    mvaddstr(5,1,"Logged in as:"); 
    sprintf(buf,"%s",my_name);
    mvaddstr(5,15,buf);

    mvaddstr(7,1,"p) Play NetHack!");
    mvaddstr(9,1,"=> ");
  } else {
    mvaddstr(5,1,"Not logged in.");
    mvaddstr(7,1,"l) Login");
    mvaddstr(8,1,"r) Register new user");
    mvaddstr(9,1,"q) Quit");
    mvaddstr(11,1,"=> ");
  }

  refresh();

  /* for retarded clients */
  loopcontrol1++;
  if (loopcontrol1 >= 20) exit (119);
}

/* ************************************************************* */

int freefile() {
  int i;

  /* free existing mem, clear existing entries */
  for(i=0;i<f_num;i++) {
    free(f_name[i]); 
    free(f_email[i]); 
    free(f_pw[i]); 
    free(f_env[i]); 
  }

  f_num = 0;
}

/* ************************************************************* */

void initncurses() {
  initscr();
  cbreak();
  echo();
  nonl();
  intrflush(stdscr, FALSE);
  keypad(stdscr, TRUE);
}

/* ************************************************************* */

void initvars() {
  memset(my_name,21,0);
  memset(my_pw,21,0);
  memset(my_email,81,0);
  memset(my_env,1025,0);
}

/* ************************************************************* */

void login() {
  char buf[1025];
  int error = 2;

  loggedin = 0;

  while (error) {
    clear();
  
    mvaddstr(1,1,VER1);
    mvaddstr(2,1,VER2);
    mvaddstr(3,1,SERVER);

    mvaddstr(5,1,"Please enter your username.");
    mvaddstr(7,1,"=> ");

    if (error==1) {
      mvaddstr(9,1,"There was a problem with your last entry.");
      move(7,4);
    }

    refresh();

    getnstr(buf,20);
    error = 1;
    if (userexist(buf)) error = 0;
  }

  strncpy(my_name,buf,20);

  clear();
  
  mvaddstr(1,1,VER1);
  mvaddstr(2,1,VER2);
  mvaddstr(3,1,SERVER);

  mvaddstr(5,1,"Please enter your password.");
  mvaddstr(7,1,"=> ");

  refresh();
  
  getnstr(buf,20);
  strncpy(my_pw,buf,20);

  if (passwordgood(my_name,my_pw)) {
    loggedin = 1; 
  }
}

/* ************************************************************* */

void newuser() {
  char buf[1024];
  int error=2;
  int i;

  loggedin = 0;

  while (error) {
    clear();
  
    mvaddstr(1,1,VER1);
    mvaddstr(2,1,VER2);
    mvaddstr(3,1,SERVER);

    mvaddstr(5,1,"Welcome new user. Please enter a username.");
    mvaddstr(6,1,"Only characters and numbers are allowed, with no spaces.");
    mvaddstr(7,1,"20 character max.");
    mvaddstr(9,1,"=> ");

    if (error==1) {
      mvaddstr(11,1,"There was a problem with your last entry.");
      move(9,4);
    }

    refresh();

    getnstr(buf,20);
    if (!userexist(buf)) error = 0;

    for(i=0;i<strlen(buf);i++) {
      if (! (((buf[i]>='a')&&(buf[i]<='z')) || ((buf[i]>='A')&&(buf[i]<='Z'))
        || ((buf[i]>='0')&&(buf[i]<='9'))) ) error = 1;
    }

    if (strlen(buf)<2) error = 1;
  }

  strncpy(my_name,buf,20);

  /* password step */

  clear();

  mvaddstr(1,1,VER1);
  mvaddstr(2,1,VER2);
  mvaddstr(3,1,SERVER);

  mvaddstr(5,1,"Please enter a password.");
  mvaddstr(6,1,"This is only trivially encoded at the server. Please use something");
  mvaddstr(7,1,"new and expect it to be relatively insecure.");
  mvaddstr(8,1,"20 character max. No ':' characters.");
  mvaddstr(10,1,"=> ");

  refresh();
  getnstr(buf,20);

  for (i=0;i<strlen(buf);i++) {
    /* we warned em */
    if(buf[i] == ':') exit(112);
  }

  strncpy(my_pw,buf,20);

  /* email step */

  clear();

  mvaddstr(1,1,VER1);
  mvaddstr(2,1,VER2);
  mvaddstr(3,1,SERVER);

  mvaddstr(5,1,"Please enter your email address.");
  mvaddstr(6,1,"This is sent _nowhere_ but will be used if you ask the sysadmin for lost");
  mvaddstr(7,1,"password help. Please use a correct one. It only benefits you.");
  mvaddstr(8,1,"80 character max. No ':' characters.");
  mvaddstr(10,1,"=> ");

  refresh();
  getnstr(buf,80);

  for (i=0;i<strlen(buf);i++) {
    /* we warned em */
    if(buf[i] == ':') exit(113);
  }

  strncpy(my_email,buf,80);

  loggedin = 1;
  writefile(1);
}

/* ************************************************************* */

int passwordgood(char *cname, char *cpw) {
  int i;

  for(i=0;i<f_num;i++) {
    if (!strncasecmp(cname,f_name[i],20)) {
      if (!strncmp(cpw,f_pw[i],20)) {
        return 1;
      } else return 0;
    }
  }

  return 0;
}

/* ************************************************************* */

int readfile(int nolock) {
  FILE *fp, *fpl;
  char buf[1200];

  memset(buf,1024,0);

  /* read new stuff */

  if (!nolock) {
    fpl = fopen("/dgl-lock","r");
    if (!fpl) exit(106);
    if (flock(fileno(fpl),LOCK_SH)) exit(114);
  }

  fp = fopen("/dgl-login","r");
  if (!fp) exit(106);

  /* once per name in the file */
  while (fgets(buf,1200,fp)) {
    char *b=buf, *n=buf;

    f_name[f_num] = (char *)calloc(22,sizeof(char));
    f_email[f_num] = (char *)calloc(82,sizeof(char));
    f_pw[f_num] = (char *)calloc(22,sizeof(char));
    f_env[f_num] = (char *)calloc(1026,sizeof(char));

    /* name field, must be valid */
    while(*b!=':') {
      if (! (((*b>='a')&&(*b<='z')) || ((*b>='A')&&(*b<='Z'))
        || ((*b>='0')&&(*b<='9'))) ) return 1;
      f_name[f_num][(b-n)] = *b;
      b++;
      if ((b-n)>=20) exit(100);
    }

    /* advance to next field */
    n = b+1;
    b = n;

    /* email field */
    while(*b!=':') {
      f_email[f_num][(b-n)] = *b;
      b++;
      if ((b-n)>=80) exit(101);
    }

    /* advance to next field */
    n = b+1;
    b = n;

    /* pw field */
    while(*b!=':') {
      f_pw[f_num][(b-n)] = *b;
      b++;
      if ((b-n)>=20) exit(102);
    }

    /* advance to next field */
    n = b+1;
    b = n;

    /* env field */
    while((*b!='\n')&&(*b!=0)&&(*b!=EOF)) {
      f_env[f_num][(b-n)] = *b;
      b++;
      if ((b-n)>=1024) exit(102);
    }

    f_num++;
    /* prevent a buffer overrun here */
    if (f_num >= MAXUSERS) exit(109);
  }

  if (!nolock) {
    flock(fileno(fpl),LOCK_UN);
    fclose(fpl);
  }
  fclose(fp);
  return 0;
}

/* ************************************************************* */

int userexist(char *cname) {
  int i;

  for(i=0;i<f_num;i++) {
    if (!strncasecmp(cname,f_name[i],20)) return 1;
  }

  return 0;
}

/* ************************************************************* */

int writefile(int requirenew) {
  FILE *fp, *fpl;
  int i=0;
  int my_done = 0;

  fpl = fopen("/dgl-lock","r");
  if (!fpl) exit(115);
  if (flock(fileno(fpl),LOCK_EX)) exit (107);

  freefile();
  readfile(1);

  fp = fopen("/dgl-login","w");
  if (!fp) exit(104);

  for(i=0;i<f_num;i++) {
    if(loggedin && !strncasecmp(my_name,f_name[i],20)) {
      if (requirenew) {
        /* this is if someone managed to register at the same time
	 * as someone else. just die. */
	exit(111);
      }
      fprintf(fp,"%s:%s:%s:%s\n",my_name,my_email,my_pw,my_env);
      my_done = 1;
    } else {
      fprintf(fp,"%s:%s:%s:%s\n",f_name[i],f_email[i],f_pw[i],f_env[i]); 
    }
  }
  if(loggedin && !my_done) { /* new entry */
      fprintf(fp,"%s:%s:%s:%s\n",my_name,my_email,my_pw,my_env);
  }

  flock(fileno(fpl),LOCK_UN);
  fclose(fp);
  fclose(fpl);
}

/* ************************************************************* */
/* ************************************************************* */
/* ************************************************************* */

int
main (int argc, char **argv)
{
  /* for chroot and program execution */
  char *argv1 = "/bin/nethack";
  char *argv2 = "-u";
  char *myargv[10];
  uid_t newuid=SHED_UID;
  gid_t newgid=SHED_GID;

  /* for login */
  char inbuf[1024]; /* always check for overruns */
  int userchoice;

  /* chroot */
  if (chroot ("/opt/nethack/nethack.dtype.org"))
    error (1, errno, "cannot change root directory");

  if (chdir ("/"))
    error (1, errno, "cannot chdir to root directory");

  /* shed privs. this is done immediately after chroot. */
  setgid(newgid);
  setuid(newuid);

  initvars();

  /* simple login routine, uses ncurses */
  if (readfile(0)) exit(110);

  initncurses();
  while ((userchoice!='p')||(!loggedin)) {
    drawmenu();
    userchoice = getch();
    if (userchoice=='q') {
      endwin();
      exit(1);
    }
    if (userchoice=='r') {
      newuser();
    }
    if (userchoice=='l') {
      login();
    }
  }

  freefile();
  endwin();

  /* launch program */
  myargv[0] = argv1;
  myargv[1] = argv2;
  myargv[2] = my_name;
  myargv[3] = 0;
  execvp ("/bin/nethack", myargv);
  error (1, errno, "cannot execute %s", argv[0]);

  exit (1);
  return 1;
}
