/* nethacklaunch.c
 *
 * (c)2001-3 M. Drew Streib <dtype@dtype.org>
 * This program is free software; you can redistribute it and/or modify 
 * it under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 2 of the License, or 
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details. 
 * 
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software 
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * See this in program action at http://dtype.org/nethack/
 *
 * This is a little wrapper for nethack (and soon other programs) that
 * will allow them to be run from a telnetd session, chroot, shed privs,
 * make a simple login, then play the game.
 *
 * By default, this thing is also statically compiled, and can thus be
 * run inside of a chroot jail itself if necessary.
 *
 * Yes, I know it is all global variables. Deal with it. The program
 * is very small.
 */

/* a request from the author: please leave some remnance of
 * 'based on dgamelaunch version xxx' in any derivative works, or
 * even keep the line the same altogether. I'm probably happy 
 * to make any changes you need. */

#define VERLINES 7							/* number of lines in this vanity text */
#define VER1 "## dgamelaunch - network console game launcher\n"
#define VER2 "## version 1.2.20\n"
#define VER3 "## \n"
#define VER4 "## (c)2001-3 M. Drew Streib. This program's source is released under the GPL.\n"
#define VER5 "## Send mail to <dtype@dtype.org> for details or a copy of the source code.\n"
#define VER6 "## ** Games on this server are recorded for in-progress viewing and playback!\n"
#define VER7 "## Server info is at http://dtype.org/nethack/"

/* ************************************************************* */
/* ************************************************************* */
/* ************************************************************* */

/* program stuff */

#include "dgamelaunch.h"
#include <stdlib.h>
#include <curses.h>
#include <crypt.h>
#include <sys/types.h>
#include <sys/file.h>						/* for flock() */
#include <sys/ioctl.h>					/* ttyrec */
#include <errno.h>
#include <time.h>
#include <dirent.h>
#include <string.h>
#include <sys/stat.h>
#include <sys/signal.h>

extern int vi_main (int argc, char **argv);
extern int ttyrec_main (char *);
extern int master;
extern int slave;
extern struct termios tt;
extern struct winsize win;

/* global variables */

int caught_sighup = 0;
int pid_game = 0;
int loggedin = 0;
char my_name[21];
char my_email[81];
char my_pw[21];
char my_env[1025];
char rcfilename[80];
char ttyrec_filename[100];

/* preallocate this mem. bad, but ohwell. is only for pointers */
/* makes a max number of users compiled in */
int f_num = 0;
char *f_name[MAXUSERS];
char *f_email[MAXUSERS];
char *f_pw[MAXUSERS];
char *f_env[MAXUSERS];

int loopcontrol1 = 0;
int loopcontrol2 = 0;

/* ************************************************************* */
/* for ttyrec */

void
ttyrec_getmaster ()
{
	(void) tcgetattr (0, &tt);
	(void) ioctl (0, TIOCGWINSZ, (char *) &win);
	if ((master = open ("/dev/ptmx", O_RDWR)) < 0)
		{
			exit (62);
		}
}

/* ************************************************************* */

void
gen_ttyrec_filename ()
{
	char spbuf[20];
	time_t rawtime;
	struct tm *ptm;

	/* append time to filename */
	time (&rawtime);
	ptm = gmtime (&rawtime);
	sprintf (spbuf, "%04i", ptm->tm_year + 1900);
	strncpy (ttyrec_filename, spbuf, 4);
	strcat (ttyrec_filename, "-");
	sprintf (spbuf, "%02i", ptm->tm_mon + 1);
	strncat (ttyrec_filename, spbuf, 2);
	strcat (ttyrec_filename, "-");
	sprintf (spbuf, "%02i", ptm->tm_mday);
	strncat (ttyrec_filename, spbuf, 2);
	strcat (ttyrec_filename, ".");
	sprintf (spbuf, "%02i", ptm->tm_hour);
	strncat (ttyrec_filename, spbuf, 2);
	strcat (ttyrec_filename, ":");
	sprintf (spbuf, "%02i", ptm->tm_min);
	strncat (ttyrec_filename, spbuf, 2);
	strcat (ttyrec_filename, ":");
	sprintf (spbuf, "%02i", ptm->tm_sec);
	strncat (ttyrec_filename, spbuf, 2);
	strcat (ttyrec_filename, ".ttyrec");
}

/* ************************************************************* */

void
gen_inprogress_lock ()
{
	char lockfile[130];
	int fd;

	strcpy (lockfile, LOC_INPROGRESSDIR);
	strcat (lockfile, my_name);
	strcat (lockfile, ":");
	strcat (lockfile, ttyrec_filename);

	fd = open (lockfile, O_WRONLY | O_CREAT);
	if (flock (fd, LOCK_EX))
		exit (68);
}

/* ************************************************************* */

void
catch_sighup ()
{
	caught_sighup = 1;
	if (pid_game)
		{
			sleep (10);
			kill (pid_game, SIGHUP);
			sleep (5);
		}
	exit (2);
}

/* ************************************************************* */

void
inprogressmenu ()
{
	int i;
	DIR *pdir;
	struct dirent *pdirent;
	int fd;
	struct stat pstat;
	char buf[20];
	char printline[130];
	char fullname[130];
	char ttyrecname[130];
	char *replacestr;
	char *games[15];
	char m_name[26], m_date[11], m_time[9];
	int m_namelen;
	time_t ctime;
	int menuchoice;


	do
		{
			clear ();
			mvaddstr (1, 1, VER1);
			mvaddstr (3, 1,
								"(Press 'q' during game playback to return to this menu.)");
			mvaddstr (4, 1, "The following games are in progress:");

			/* clean old games and list good ones */
			i = 0;
			pdir = opendir (LOC_INPROGRESSDIR);
			if (!pdir)
				exit (140);

			while ((pdirent = readdir (pdir)) && (i <= 15))
				{
					strcpy (fullname, LOC_INPROGRESSDIR);
					strncat (fullname, pdirent->d_name, 100);

					fd = 0;
					fd = open (fullname, O_RDONLY);
					if ((fd > 0) && flock (fd, LOCK_EX | LOCK_NB))
						{

							/* stat to check idle status */
							strcpy (ttyrecname, LOC_TTYRECDIR);
							strncat (ttyrecname, pdirent->d_name, 100);
							replacestr = strstr (ttyrecname, ":");
							if (!replacestr)
								exit (145);
							replacestr[0] = '/';
							if (!stat (ttyrecname, &pstat))
								{
									games[i] = pdirent->d_name;

									memset (m_name, 0, 26);
									memset (m_date, 0, 11);
									memset (m_time, 0, 9);
									m_namelen =
										replacestr - ttyrecname - strlen (LOC_TTYRECDIR);
									strncpy (m_name, pdirent->d_name, m_namelen);
									strncpy (m_date, replacestr + 1, 10);
									strncpy (m_time, replacestr + 12, 8);

									sprintf (printline, "%c) %-15s %s %s (%im %is idle)",
													 i + 97, m_name, m_date, m_time,
													 (time (&ctime) - pstat.st_mtime) / 60,
													 (time (&ctime) - pstat.st_mtime) % 60);
									mvaddstr (6 + i++, 1, printline);
								}
						}
					else
						{
							unlink (fullname);
						}
					flock (fd, LOCK_UN | LOCK_NB);
					close (fd);
				}

			mvaddstr (23, 1, "Watch which game? (r to refresh, q to quit) => ");
			refresh ();

			menuchoice = getch ();

			if ((menuchoice - 97) >= 0 && (menuchoice - 97) < i)
				{
					/* valid choice has been made */
					strcpy (ttyrecname, LOC_TTYRECDIR);
					strncat (ttyrecname, games[menuchoice - 97], 100);
					replacestr = strstr (ttyrecname, ":");
					if (!replacestr)
						exit (145);
					replacestr[0] = '/';

					clear ();
					refresh ();
					endwin ();
					ttyplay_main (ttyrecname, 1);
				}

			closedir (pdir);
		}
	while (menuchoice != 'q');
}

/* ************************************************************* */

void
changepw ()
{
	char buf[21];
	int i;

	if (!loggedin)
		return;
	clear ();

	mvaddstr (1, 1, VER1);

	mvaddstr (5, 1,
						"Please enter a new password. Remember that this is sent over the net");
	mvaddstr (6, 1,
						"in plaintext, so make it something new and expect it to be relatively");
	mvaddstr (7, 1, "insecure.");
	mvaddstr (8, 1, "20 character max. No ':' characters.");
	mvaddstr (10, 1, "=> ");

	refresh ();
	getnstr (buf, 20);

	for (i = 0; i < strlen (buf); i++)
		{
			/* we warned em */
			if (buf[i] == ':')
				exit (112);
		}

	strncpy (my_pw, crypt (buf, buf), 14);
	writefile (0);
}

/* ************************************************************* */

void
drawmenu ()
{
	char buf[80];

	clear ();

	mvaddstr (1, 1, VER1);
	mvaddstr (2, 1, VER2);
	mvaddstr (3, 1, VER3);
	mvaddstr (4, 1, VER4);
	mvaddstr (5, 1, VER5);
	mvaddstr (6, 1, VER6);
	mvaddstr (7, 1, VER7);

	if (loggedin)
		{
			mvaddstr (VERLINES + 2, 1, "Logged in as:");
			sprintf (buf, "%s", my_name);
			mvaddstr (VERLINES + 2, 15, buf);

			mvaddstr (VERLINES + 4, 1, "c) Change password");
			mvaddstr (VERLINES + 5, 1, "o) Edit option file (requires vi use)");
			mvaddstr (VERLINES + 6, 1, "w) Watch games in progress");
			mvaddstr (VERLINES + 7, 1, "p) Play nethack!");
			mvaddstr (VERLINES + 8, 1, "q) Quit");
			mvaddstr (VERLINES + 10, 1, "=> ");
		}
	else
		{
			mvaddstr (VERLINES + 2, 1, "Not logged in.");
			mvaddstr (VERLINES + 4, 1, "l) Login");
			mvaddstr (VERLINES + 5, 1, "r) Register new user");
			mvaddstr (VERLINES + 6, 1, "w) Watch games in progress");
			mvaddstr (VERLINES + 7, 1, "q) Quit");
			mvaddstr (VERLINES + 9, 1, "=> ");
		}

	refresh ();

	/* for retarded clients */
	loopcontrol1++;
	if (loopcontrol1 >= 20)
		exit (119);
}

/* ************************************************************* */

int
freefile ()
{
	int i;

	/* free existing mem, clear existing entries */
	for (i = 0; i < f_num; i++)
		{
			free (f_name[i]);
			free (f_email[i]);
			free (f_pw[i]);
			free (f_env[i]);
		}

	f_num = 0;
}

/* ************************************************************* */

void
initncurses ()
{
	initscr ();
	cbreak ();
	echo ();
	nonl ();
	intrflush (stdscr, FALSE);
	keypad (stdscr, TRUE);
}

/* ************************************************************* */

void
initvars ()
{
	memset (my_name, 21, 0);
	memset (my_pw, 21, 0);
	memset (my_email, 81, 0);
	memset (my_env, 1025, 0);
}

/* ************************************************************* */

void
login ()
{
	char buf[1025];
	int error = 2;

	loggedin = 0;

	while (error)
		{
			clear ();

			mvaddstr (1, 1, VER1);

			mvaddstr (5, 1, "Please enter your username.");
			mvaddstr (7, 1, "=> ");

			if (error == 1)
				{
					mvaddstr (9, 1, "There was a problem with your last entry.");
					move (7, 4);
				}

			refresh ();

			getnstr (buf, 20);
			error = 1;
			if (userexist (buf))
				error = 0;
		}

	strncpy (my_name, buf, 20);

	clear ();

	mvaddstr (1, 1, VER1);

	mvaddstr (5, 1, "Please enter your password.");
	mvaddstr (7, 1, "=> ");

	refresh ();

	noecho ();
	getnstr (buf, 20);
	echo ();
	strncpy (my_pw, buf, 20);

	if (passwordgood (my_name, my_pw))
		{
			loggedin = 1;
			strcpy (rcfilename, LOC_DGLDIR);
			strcat (rcfilename, my_name);
			strcat (rcfilename, ".nethackrc");
		}
}

/* ************************************************************* */

void
newuser ()
{
	char buf[1024];
	int error = 2;
	int i;

	loggedin = 0;

	while (error)
		{
			clear ();

			mvaddstr (1, 1, VER1);

			mvaddstr (5, 1, "Welcome new user. Please enter a username.");
			mvaddstr (6, 1,
								"Only characters and numbers are allowed, with no spaces.");
			mvaddstr (7, 1, "20 character max.");
			mvaddstr (9, 1, "=> ");

			if (error == 1)
				{
					mvaddstr (11, 1, "There was a problem with your last entry.");
					move (9, 4);
				}

			refresh ();

			getnstr (buf, 20);
			if (!userexist (buf))
				error = 0;

			for (i = 0; i < strlen (buf); i++)
				{
					if (!
							(((buf[i] >= 'a') && (buf[i] <= 'z'))
							 || ((buf[i] >= 'A') && (buf[i] <= 'Z')) || ((buf[i] >= '0')
																													 && (buf[i] <=
																															 '9'))))
						error = 1;
				}

			if (strlen (buf) < 2)
				error = 1;
		}

	strncpy (my_name, buf, 20);

	/* password step */

	clear ();

	mvaddstr (1, 1, VER1);

	mvaddstr (5, 1, "Please enter a password.");
	mvaddstr (6, 1,
						"This is only trivially encoded at the server. Please use something");
	mvaddstr (7, 1, "new and expect it to be relatively insecure.");
	mvaddstr (8, 1, "20 character max. No ':' characters.");
	mvaddstr (10, 1, "=> ");

	refresh ();
	getnstr (buf, 20);

	for (i = 0; i < strlen (buf); i++)
		{
			/* we warned em */
			if (buf[i] == ':')
				exit (112);
		}

	strncpy (my_pw, crypt (buf, buf), 14);

	/* email step */

	clear ();

	mvaddstr (1, 1, VER1);

	mvaddstr (5, 1, "Please enter your email address.");
	mvaddstr (6, 1,
						"This is sent _nowhere_ but will be used if you ask the sysadmin for lost");
	mvaddstr (7, 1,
						"password help. Please use a correct one. It only benefits you.");
	mvaddstr (8, 1, "80 character max. No ':' characters.");
	mvaddstr (10, 1, "=> ");

	refresh ();
	getnstr (buf, 80);

	for (i = 0; i < strlen (buf); i++)
		{
			/* we warned em */
			if (buf[i] == ':')
				exit (113);
		}

	strncpy (my_email, buf, 80);

	loggedin = 1;
	writefile (1);
}

/* ************************************************************* */

int
passwordgood (char *cname, char *cpw)
{
	int i;

	for (i = 0; i < f_num; i++)
		{
			if (!strncasecmp (cname, f_name[i], 20))
				{

					/* first check crypt() version, then plaintext */
					if (!strncmp (crypt (cpw, cpw), f_pw[i], 13))
						return 1;
					if (!strncmp (cpw, f_pw[i], 20))
						return 1;
				}
		}

	return 0;
}

/* ************************************************************* */

int
readfile (int nolock)
{
	FILE *fp, *fpl;
	char buf[1200];

	memset (buf, 1024, 0);

	/* read new stuff */

	if (!nolock)
		{
			fpl = fopen ("/dgl-lock", "r");
			if (!fpl)
				exit (106);
			if (flock (fileno (fpl), LOCK_SH))
				exit (114);
		}

	fp = fopen ("/dgl-login", "r");
	if (!fp)
		exit (106);

	/* once per name in the file */
	while (fgets (buf, 1200, fp))
		{
			char *b = buf, *n = buf;

			f_name[f_num] = (char *) calloc (22, sizeof (char));
			f_email[f_num] = (char *) calloc (82, sizeof (char));
			f_pw[f_num] = (char *) calloc (22, sizeof (char));
			f_env[f_num] = (char *) calloc (1026, sizeof (char));

			/* name field, must be valid */
			while (*b != ':')
				{
					if (!(((*b >= 'a') && (*b <= 'z')) || ((*b >= 'A') && (*b <= 'Z'))
								|| ((*b >= '0') && (*b <= '9'))))
						return 1;
					f_name[f_num][(b - n)] = *b;
					b++;
					if ((b - n) >= 21)
						exit (100);
				}

			/* advance to next field */
			n = b + 1;
			b = n;

			/* email field */
			while (*b != ':')
				{
					f_email[f_num][(b - n)] = *b;
					b++;
					if ((b - n) >= 80)
						exit (101);
				}

			/* advance to next field */
			n = b + 1;
			b = n;

			/* pw field */
			while (*b != ':')
				{
					f_pw[f_num][(b - n)] = *b;
					b++;
					if ((b - n) >= 20)
						exit (102);
				}

			/* advance to next field */
			n = b + 1;
			b = n;

			/* env field */
			while ((*b != '\n') && (*b != 0) && (*b != EOF))
				{
					f_env[f_num][(b - n)] = *b;
					b++;
					if ((b - n) >= 1024)
						exit (102);
				}

			f_num++;
			/* prevent a buffer overrun here */
			if (f_num >= MAXUSERS)
				exit (109);
		}

	if (!nolock)
		{
			flock (fileno (fpl), LOCK_UN);
			fclose (fpl);
		}
	fclose (fp);
	return 0;
}

/* ************************************************************* */

int
userexist (char *cname)
{
	int i;

	for (i = 0; i < f_num; i++)
		{
			if (!strncasecmp (cname, f_name[i], 20))
				return 1;
		}

	return 0;
}

/* ************************************************************* */

int
editoptions ()
{
	FILE *rcfile;
	char *myargv[3];

	rcfile = fopen (rcfilename, "r");
	printf (" read");
	if (!rcfile)
		{
			rcfile = fopen (rcfilename, "w");
			if (!rcfile)
				{
					exit (76);
				}

			fprintf (rcfile,
							 "# This is your personal nethackrc file. It is in the same format as default\n");
			fprintf (rcfile,
							 "# nethack options files. Optionally uncomment the following lines as a\n");
			fprintf (rcfile,
							 "# starter. This editor is vi-like. Type ESC a couple times, then ':q!'\n");
			fprintf (rcfile, "# (without quotes) to exit if you get stuck.\n");
			fprintf (rcfile, "#\n");
			fprintf (rcfile,
							 "# See the following link for more options information:\n");
			fprintf (rcfile,
							 "# http://www.nethack.org/v341/Guidebook.html#_TOCentry_40\n");
			fprintf (rcfile, "#\n");
			fprintf (rcfile, "# OPTIONS=showexp,showscore,time\n");
			fprintf (rcfile, "# OPTIONS=!autopickup,color\n");
			fprintf (rcfile,
							 "# OPTIONS=catname:catfoo,dogname:dogfoo,horsename:horsefoo\n");
			fprintf (rcfile, "#\n");
			fprintf (rcfile,
							 "# The following lines are necessary if you want to use the menucolors option.\n");
			fprintf (rcfile, "# OPTIONS=menucolors\n");
			fprintf (rcfile, "# MENUCOLOR=\" blessed \"=green\n");
			fprintf (rcfile, "# MENUCOLOR=\" holy \"=green\n");
			fprintf (rcfile, "# MENUCOLOR=\" uncursed \"=yellow\n");
			fprintf (rcfile, "# MENUCOLOR=\" cursed \"=red\n");
			fprintf (rcfile, "# MENUCOLOR=\" unholy \"=red\n");
			fprintf (rcfile,
							 "# MENUCOLOR=\" cursed .* (being worn)\"=orange&underline\n");
		}

	fclose (rcfile);

	/* use virus to edit */

	myargv[0] = "";
	myargv[1] = rcfilename;
	myargv[2] = 0;

	endwin ();
	vi_main (2, myargv);
	refresh ();
}

/* ************************************************************* */

int
writefile (int requirenew)
{
	FILE *fp, *fpl;
	int i = 0;
	int my_done = 0;

	fpl = fopen ("/dgl-lock", "r");
	if (!fpl)
		exit (115);
	if (flock (fileno (fpl), LOCK_EX))
		exit (107);

	freefile ();
	readfile (1);

	fp = fopen ("/dgl-login", "w");
	if (!fp)
		exit (104);

	for (i = 0; i < f_num; i++)
		{
			if (loggedin && !strncasecmp (my_name, f_name[i], 20))
				{
					if (requirenew)
						{
							/* this is if someone managed to register at the same time
							 * as someone else. just die. */
							exit (111);
						}
					fprintf (fp, "%s:%s:%s:%s\n", my_name, my_email, my_pw, my_env);
					my_done = 1;
				}
			else
				{
					fprintf (fp, "%s:%s:%s:%s\n", f_name[i], f_email[i], f_pw[i],
									 f_env[i]);
				}
		}
	if (loggedin && !my_done)
		{														/* new entry */
			fprintf (fp, "%s:%s:%s:%s\n", my_name, my_email, my_pw, my_env);
		}

	flock (fileno (fpl), LOCK_UN);
	fclose (fp);
	fclose (fpl);
}

/* ************************************************************* */
/* ************************************************************* */
/* ************************************************************* */

int
main (int argc, char **argv)
{
	/* for chroot and program execution */
	char *argv1 = LOC_NETHACK;
	char *argv2 = "-u";
	char *myargv[10];
	uid_t newuid = SHED_UID;
	gid_t newgid = SHED_GID;
	char atrcfilename[81];

	/* for login */
	char inbuf[1024];							/* always check for overruns */
	int userchoice;

	/* signal handlers */
	signal (SIGHUP, catch_sighup);

	/* get master tty just before chroot (lives in /dev) */
	ttyrec_getmaster ();
	grantpt (master);
	unlockpt (master);
	if ((slave = open ((const char *) ptsname (master), O_RDWR)) < 0)
		{
			exit (65);
		}


	/* chroot */
	if (chroot (LOC_CHROOT))
		error (1, errno, "cannot change root directory");

	if (chdir ("/"))
		error (1, errno, "cannot chdir to root directory");

	/* shed privs. this is done immediately after chroot. */
	setgid (newgid);
	setuid (newuid);

	initvars ();

	/* simple login routine, uses ncurses */
	if (readfile (0))
		exit (110);

	initncurses ();
	while ((userchoice != 'p') | (!loggedin))
		{
			drawmenu ();
			userchoice = getch ();
			if (userchoice == 'c')
				{
					changepw ();
				}
			if (userchoice == 'w')
				{
					inprogressmenu ();
				}
			if (userchoice == 'o' && loggedin)
				{
					editoptions ();
				}
			if (userchoice == 'q')
				{
					endwin ();
					exit (1);
				}
			if (userchoice == 'r')
				{
					newuser ();
				}
			if (userchoice == 'l')
				{
					login ();
				}
		}

	freefile ();
	endwin ();

	/* environment */
	strcpy (atrcfilename, "@");
	strcat (atrcfilename, rcfilename);
	setenv ("NETHACKOPTIONS", atrcfilename, 1);

	/* lock */
	gen_ttyrec_filename ();
	gen_inprogress_lock ();

	/* launch program */
	ttyrec_main (my_name);
	/* myargv[0] = argv1;
	   myargv[1] = argv2;
	   myargv[2] = my_name;
	   myargv[3] = 0;
	   execvp (LOC_NETHACK, myargv);
	   error (1, errno, "cannot execute %s", argv[0]);
	 */

	exit (1);
	return 1;
}
